/* Source Code for xSPI Profile 1.0 FLASH  Low Level Driver */

/*
*  DISCLAIMER:
*
*      THIS SOFTWARE, SOURCE CODE AND ASSOCIATED MATERIALS INCLUDING BUT NOT LIMITED TO TUTORIALS,
*      GUIDES AND COMMENTARY PROVIDED WITH THIS EXERCISE ARE ONLY DESIGNED FOR REFERENCE PURPOSES
*      TO GIVE AN EXAMPLE TO LICENSEE FOR THEIR OWN NECESSARY DEVELOPMENT OF THEIR OWN SOFTWARE AND/OR
*      APPLICATION. IT IS NOT DESIGNED FOR ANY SPECIAL PURPOSE, SERIAL PRODUCTION OR USE IN MEDICAL,
*      MILITARY, AIR CRAFT, AVIATION, SPACE OF LIFE SUPPORT EQUIPMENT.
*
*      TO THE EXTENT PERMITTED BY LAW, THE EXERCISE SOFTWARE AND/OR SOURCE CODE AND/OR AND ASSOCIATED
*      MATERIALS IS PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND AND ONLY FOR REFERENCE PURPOSES.
*
*      SYNAPTIC LABORATORIES LTD. MAKES NO WARRANTIES, EITHER EXPRESS OR IMPLIED, WITH RESPECT TO THE
*      LICENSED SOFTWARE AND/OR SOURCE CODE AND/OR ASSOCIATED MATERIALS, CONFIDENTIAL INFORMATION AND
*      DOCUMENTATION PROVIDED HEREUNDER.
*
*      SYNAPTIC LABORATORIES LTD. SPECIFICALLY DISCLAIMS THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
*      FITNESS FOR A PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT OF ANY INTELLECTUAL
*      PROPERTY RIGHT OF ANY THIRD PARTY WITH REGARD TO THE SOFTWARE, DOCUMENTATION (SCHEMATICS ETC.),
*      SOURCE CODE AND ASSOCIATED MATERIALS, CONFIDENTIAL INFORMATION AND DOCUMENTATION.
*
*      ANY USE, COMPILATION AND TESTING OF THE SOFTWARE AND/OR SOURCE CODE IS AT LICENSEE`S OWN RISK
*      AND LICENSEE IS OBLIGED TO CONDUCT EXTENSIVE TESTS TO AVOID ANY ERRORS AND FAILURE IN THE
*      COMPILED SOURCE CODE, DOCUMENTATION (SCHEMATICS ETC.) AND THE HEREFROM GENERATED SOFTWARE
*      OF LICENSEE.
*
*      EXCEPT FOR WILFULL INTENT SYNAPTIC LABORATORIES LTD. SHALL IN NO EVENT BE ENTITLED TO OR LIABLE
*      FOR ANY INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND OR NATURE, INCLUDING,
*      WITHOUT LIMITATION, BUSINESS INTERRUPTION COSTS, LOSS OF PROFIT OR REVENUE, LOSS OF DATA,
*      PROMOTIONAL OR MANUFACTURING EXPENSES, OVERHEAD, COSTS OR EXPENSES ASSOCIATED WITH WARRANTY
*      OR INTELLECTUAL PROPERTY INFRINGEMENT CLAIMS, INJURY TO REPUTATION OR LOSS OF CUSTOMERS.
*
*/


#include "sll_drvr_nor_flash_002.h"


/* private functions */


/******************************************************************************
* 
* sll_nor_flash_dev_int - initialise flash handle
*
*
* RETURNS: void
*
* ERRNO: 
*/
void sll_nor_flash_dev_int
(
xSPI_MBMC_Info    *xSPI_MBMC_dev, /* Flash device*/
sll_xspi_flash_dev *fd            /* Flash device*/
)
{

  fd->csr_base          = ((uint32_t)(xSPI_MBMC_dev->csr_base));
  fd->data_base         = ((uint32_t)(xSPI_MBMC_dev->dev0_mem_base));
  fd->data_end          = ((uint32_t)(xSPI_MBMC_dev->dev0_mem_base) + (uint32_t)(xSPI_MBMC_dev->dev0_mem_size));
  fd->size_in_bytes     = ((uint32_t)(xSPI_MBMC_dev->dev0_mem_size));
  fd->number_of_sectors = ((uint32_t)(xSPI_MBMC_dev->dev0_mem_size)/fd->sector_size);
	
}


/******************************************************************************
* 
* sll_nor_flash_drvr_ResetCmd - issues a Software Reset command to the flash device
*
*
* RETURNS: void
*
* ERRNO: 
*/

void sll_nor_flash_drvr_ResetCmd
(
sll_xspi_flash_dev *fd   /* control port base address in system */
)
{       
  /* Issue Software RESET command */
    sll_cs0_xspi_p1_format_1a_write(fd->csr_base, fd->dev.reset_cmd);
}



/******************************************************************************
* 
* sll_nor_flash_drvr_GetDeviceId - Get Flash device ID information
*
* RETURNS:  Flash ID information
*
*/
uint32_t sll_nor_flash_drvr_GetDeviceId
(
sll_xspi_flash_dev *fd   /* control port base address in system */
)
{
  volatile uint32_t id;

  id = sll_xspi_mc_reg_read (fd->csr_base, fd->dev.id_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_reg_rd_cmd);

  return(id);
}


/******************************************************************************
* 
* sll_nor_flash_drvr_GetFlagStatus - Get Flash Flag Status Register
*
* RETURNS: Flash Flag status register
*
*/
uint32_t sll_nor_flash_drvr_GetFlagStatus
(
sll_xspi_flash_dev *fd   /* control port base address in system */
)
{

  volatile uint32_t status_reg;

  status_reg = sll_xspi_mc_reg_read (fd->csr_base, fd->dev.flag_status_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_status_rd_cmd);

  return(status_reg);
}


/******************************************************************************
* 
* sll_nor_flash_drvr_GetStatus - Get Flash Flag Status Register
*
* RETURNS: Flash status register
*
*/
uint32_t sll_nor_flash_drvr_GetStatus
(
sll_xspi_flash_dev *fd   /* control port base address in system */
)
{

  volatile uint32_t status_reg;

  status_reg = sll_xspi_mc_reg_read (fd->csr_base, fd->dev.status_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_status_rd_cmd);

  return(status_reg);
}

/******************************************************************************
* 
* sll_nor_flash_drvr_ReadAnyReg - Get Any regiseter
*
* RETURNS: register content
*
*/
uint32_t sll_nor_flash_drvr_ReadAnyReg
(
sll_xspi_flash_dev *fd ,   /* control port base address in system */
uint32_t offset   
)
{

  volatile uint32_t status_reg;
  
  status_reg = sll_xspi_mc_reg_read (fd->csr_base , fd->dev.any_reg_rd_cmd, offset,  fd->dev.xspi_mbmc_reg_rd_cmd);

  return(status_reg);
}



/******************************************************************************
* 
* sll_nor_flash_drvr_Poll - Polls flash status device 
*
*
* RETURNS: value of status register
*
*/
uint32_t sll_nor_flash_drvr_Poll
(
sll_xspi_flash_dev *fd,          /* control port base address in system */
uint32_t offset             /* address offset from base address (byte aligned) */
)
{       
  uint32_t timeout_counter     = 0xFFFFFF; //set timeout vaue
  volatile uint32_t status_reg = 0xFFFF;

  do
  {
    timeout_counter--;

    if (fd->dev.use_flg_status_register > 0) {
      status_reg = sll_xspi_mc_reg_read (fd->csr_base, fd->dev.flag_status_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_status_rd_cmd);
    
      /* Check whether Device is Ready */
      if( (status_reg & fd->dev.flash_dev_flag_ready_mask) == fd->dev.flash_dev_flag_ready  )  
        break;
    } else {
      status_reg = sll_xspi_mc_reg_read (fd->csr_base , fd->dev.status_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_status_rd_cmd);
       
      /* Check whether Device is Ready */
     if( (status_reg & fd->dev.flash_dev_ready_mask) == fd->dev.flash_dev_ready  ) 
        break;
    }
      
   //wait a bit before next Poll       
   SLL_DelayMicroseconds(20)   ;

  } while(timeout_counter);
  
  return( status_reg );          /* retrun the status reg. */
}



/******************************************************************************
* 
* sll_nor_flash_drvr_ReadCFI - Gets CFI data
*
*
* RETURNS: pointer to buffer
*
*/

                  
void sll_nor_flash_drvr_ReadCFI
(
sll_xspi_flash_dev *fd,          /* control port base address in system */
uint32_t *data_buf          /* buffer containing CFI data */

)
{       
  sll_xspi_mc_reg_read_blk (fd->csr_base, fd->dev.sfpd_rd_cmd, 0x00000000,  fd->dev.xspi_mbmc_sfpd_rd_cmd, 128, data_buf);

};


                  

/******************************************************************************
* 
* sll_nor_flash_drvr_WriteAnyReg - Program Any regiseter
*
* RETURNS: None
*
*/
void sll_nor_flash_drvr_WriteAnyReg
(
sll_xspi_flash_dev *fd ,   /* control port base address in system */
uint32_t offset   ,   /* register address   */
uint32_t data         /* Data to be written */
)
{

  sll_cs0_xspi_p1_format_1d_write(fd->csr_base, fd->dev.any_reg_wr_cmd, offset, data);

}


/******************************************************************************
* 
* sll_nor_flash_drvr_WriteAnyReg_x16 - Program Any regiseter (16 bit value)
*
* RETURNS: None
*
*/
void sll_nor_flash_drvr_WriteAnyReg_x16
(
sll_xspi_flash_dev *fd ,   /* control port base address in system */
uint32_t offset   ,   /* register address */
uint16_t data         /* Data to be written (16 bit value) */
)
{
  sll_cs0_xspi_p1_format_1d_write_x16(fd->csr_base, fd->dev.any_reg_wr_cmd, offset, data);

}


/******************************************************************************
* 
* sll_nor_flash_drvr_WriteBufferProgramOp - Performs a Write Buffer Programming Operation.
*
* RETURNS: NOR_DEV_STATUS
*/
NOR_DEV_STATUS sll_nor_flash_drvr_WriteBufferProgramOp
(
sll_xspi_flash_dev *fd,   /* control port base address in system     */
uint32_t   offset,     /* address offset from base address  (byte aligned) */
uint32_t   byte_count, /* number of bytes to program        */
uint32_t *data_buf     /* buffer containing data to program */
)
{
  uint32_t status_reg;

  /* don't try with a count of zero */
  if (byte_count == 0) 
  {
    return(NOR_DEV_EMPTY_COUNT);
  }
  
  /* don't try if count is greater than supported buffer size */
  if (byte_count > 512) 
  {
    return(NOR_DEV_WRITE_BUFFER_OVERFLOW);
  }


  /* Issue unlock sequence command WREN*/
  sll_cs0_xspi_p1_format_1a_write(fd->csr_base, fd->dev.wren_cmd);
 
  /* Write Write Buffer Load Command */
  sll_cs0_xspi_p1_format_1d_write_blk(fd->csr_base, fd->dev.bulk_wr_cmd, offset, (byte_count/4), data_buf);

  /* Get Status */
  status_reg = sll_nor_flash_drvr_Poll(fd, 0x0000);

 if (fd->dev.use_flg_status_register>0 || fd->dev.use_flg2_status_register >0) {
  /* Get Flag Status */
  status_reg = sll_nor_flash_drvr_GetFlagStatus(fd);

  /* Check Lock Flag  */
  if( status_reg & fd->dev.flash_dev_flag_sec_lock_mask )
    return( NOR_DEV_LOCKED_SECTOR );    /* sector locked */

  /* Check Program Flag  */
  if( (status_reg & fd->dev.flash_dev_flag_program_mask) == fd->dev.flash_dev_flag_program_mask )
    return( NOR_DEV_PROGRAM_ERROR );    /* program error */
 } else {
  /* Check Program Flag  */
  if( (status_reg & fd->dev.flash_dev_program_mask) == fd->dev.flash_dev_program_mask )
    return( NOR_DEV_PROGRAM_ERROR );    /* program error */

}

  return( NOR_DEV_IS_IDLE );         /* program complete */
}


/******************************************************************************
* 
* sll_nor_flash_drvr_ProgramOp - Performs a standard Programming Operation to a single location.
*
* RETURNS: NOR_DEV_STATUS
*/
NOR_DEV_STATUS sll_nor_flash_drvr_ProgramOp
(
sll_xspi_flash_dev *fd,      /* device base address is system */
uint32_t offset,        /* address offset from base address (byte aligned) */
uint32_t write_data     /* variable containing data to program */
)
{   
 NOR_DEV_STATUS status;
 status = sll_nor_flash_drvr_WriteBufferProgramOp(fd, offset, 4, &write_data );
 return(status);
}


/******************************************************************************
* 
* sll_nor_flash_drvr_Sector4kEraseOp - Performs a 4K Sector Erase Operation
*
*
* RETURNS: NOR_DEV_STATUS
*
* ERRNO: 
*/

NOR_DEV_STATUS sll_nor_flash_drvr_Sector4kEraseOp
(
sll_xspi_flash_dev *fd,    /* device base address is system */
uint32_t offset       /* address offset from base address (byte aligned) */
)
{
  uint32_t         status_reg;

 /* Issue unlock sequence command */
  sll_cs0_xspi_p1_format_1a_write(fd->csr_base, fd->dev.wren_cmd);

 /* Issue erase command */
  sll_cs0_xspi_p1_format_1c_write(fd->csr_base, fd->dev.sector_4k_erase_cmd, offset);

 /* Get Status */
  status_reg = sll_nor_flash_drvr_Poll(fd, 0x0000 );

 if (fd->dev.use_flg_status_register > 0 || fd->dev.use_flg2_status_register > 0) {

  /* Get Flag Status */
  status_reg = sll_nor_flash_drvr_GetFlagStatus(fd);

 /* Check LOCK Error */  
  if( status_reg & fd->dev.flash_dev_flag_sec_lock_mask )
    return( NOR_DEV_LOCKED_SECTOR );    /* sector locked */

 /*Check Erase Error */  
  if( (status_reg & fd->dev.flash_dev_flag_erase_mask) == fd->dev.flash_dev_flag_erase_mask )
    return( NOR_DEV_ERASE_ERROR );    /* erase error */
 } else {
 /* Check Erase Error */  
  if( (status_reg & fd->dev.flash_dev_erase_mask) == fd->dev.flash_dev_erase_mask )
    return( NOR_DEV_ERASE_ERROR );    /* erase error */

}
      
  return( NOR_DEV_IS_IDLE );         /* erease complete */
}


/******************************************************************************
* 
* sll_nor_flash_drvr_SectorEraseOp - Performs a Sector Erase Operation 
*                                   (based on device sector size)
*
* RETURNS: NOR_DEV_STATUS
*
* ERRNO: 
*/


NOR_DEV_STATUS sll_nor_flash_drvr_SectorEraseOp
(
sll_xspi_flash_dev *fd,    /* device base address is system */
uint32_t offset        /* address offset from base address (byte aligned) */
)
{
  uint32_t         status_reg;

 /* Issue unlock sequence command */
  sll_cs0_xspi_p1_format_1a_write(fd->csr_base, fd->dev.wren_cmd);

 /* Issue erase command */
  sll_cs0_xspi_p1_format_1c_write(fd->csr_base, fd->dev.sector_erase_cmd, offset);

 
 /* Get Status */
  status_reg = sll_nor_flash_drvr_Poll(fd, 0x0000 );

 if (fd->dev.use_flg_status_register > 0  || fd->dev.use_flg2_status_register > 0) {

  /* Get Flag Status */
  status_reg = sll_nor_flash_drvr_GetFlagStatus(fd);
  
  /* Check Lock Error */  
  if( status_reg & fd->dev.flash_dev_flag_sec_lock_mask )
    return( NOR_DEV_LOCKED_SECTOR );    /* sector is locked */

  /* Check Erase Error */  
  if( (status_reg & fd->dev.flash_dev_flag_erase_mask) == fd->dev.flash_dev_flag_erase_mask )
    return( NOR_DEV_ERASE_ERROR );    /* erase error */
 } else {
  /* Check Erase Error */  
  if( (status_reg & fd->dev.flash_dev_erase_mask) == fd->dev.flash_dev_erase_mask )
    return( NOR_DEV_ERASE_ERROR );    /* erase error */

}
      
  return( NOR_DEV_IS_IDLE );         /* erease complete */
}

/******************************************************************************
* 
* sll_nor_flash_drvr_memcpy :  Performs program operation on a chunk of data
*                             Divides incoming data into a number of chunks
*                             Issues a write Buffer Program for each data chunk
*
* RETURNS: NOR_DEV_STATUS
*
*/
NOR_DEV_STATUS sll_nor_flash_drvr_memcpy
(
sll_xspi_flash_dev *fd,   /* device base address is system */
uint32_t   offset,          /* address offset from base address (byte aligned) */
uint32_t   byte_count,      /* number of bytes to program */
uint32_t *data_buf          /* buffer containing data to program */
)
{
	uint32_t  flash_page_size = fd->page_size;
  uint32_t  current_count   = byte_count;
  uint32_t  mask            = flash_page_size - 1;
  uint32_t  initial_count   = byte_count;
  NOR_DEV_STATUS status     = NOR_DEV_IS_IDLE;

  if (offset & mask)
  {
    /* check whether count is greater than buffer size */
    if (byte_count < (flash_page_size - (offset & mask)) )
      initial_count = byte_count; 
    else
      initial_count = flash_page_size - (offset & mask);

    /* program the first few to get write buffer aligned */
    status = sll_nor_flash_drvr_WriteBufferProgramOp(fd, offset, initial_count, data_buf);
    if (status != NOR_DEV_IS_IDLE) 
    {
     return(status);
    }

    offset        += initial_count; /* adjust pointers and counter */
    current_count -= initial_count;
    data_buf      += (initial_count/4);
    if (current_count == 0)
    {
     return(status);
    }
  }

  /* program next chunks */
  while(current_count >= flash_page_size) 
  {
    status = sll_nor_flash_drvr_WriteBufferProgramOp(fd, offset, flash_page_size, data_buf);
    if (status != NOR_DEV_IS_IDLE)
    {
      return(status);
    }

    offset        +=  flash_page_size; /* adjust pointers and counter */
    current_count -=  flash_page_size;
    data_buf      += (flash_page_size/4);
  }
  if (current_count == 0)
  {
    return(status);
  }
  
  /* Write the remaining data if any*/
  status = sll_nor_flash_drvr_WriteBufferProgramOp(fd, offset, current_count, data_buf);
  return(status);
}

